<?php

namespace App\Http\Controllers\api;

use Carbon\Carbon;
use App\Models\Message;
use App\Models\Conversation;
use Illuminate\Http\Request;
use GuzzleHttp\Promise\Create;
use Illuminate\Support\Facades\Log;
use App\Http\Resources\MessageCollection;
use App\Http\Controllers\Controller;

class ChatController extends Controller
{
    // 1. CORRECCIÓN: Se agrega el argumento $userId = 1
    public function createConversation($userId = 1)
    {
        $conversation = Conversation::create([
            'user_id' => $userId, // Se usa la variable pasada por argumento
            'start_date' => now(),
            'last_activity' => now(),
            'status' => 'open',
            'is_active' => 1,
            'status_bot' => 1
        ]);

        return $conversation;
    }

    public function createMessage($conversation, $prompt)
    {
        $message = Message::create([
            'conversation_id' => $conversation->id,
            'api_id' => time() . now(),
            'choices' => [['text' => $prompt]],
            'usage' => '',
            'date_of_issue' => now(),
            'status' => 1,
            'emisor' => 'User'
        ]);

        return $message;
    }

    public function getConversation($id)
    {
        try {
            $conversation = Conversation::with(['messages'])->where('user_id', $id)->firstOrFail();
            return response()->json([
                'data' => new MessageCollection($conversation->messages),
                'status' => true
            ], 200);
        } catch (\Throwable $th) {
            return response()->json([
                'message' => $th->getMessage(),
                'status' => false
            ], 500);
        }
    }

    // 2. CORRECCIÓN: Se eliminó la función 'chat' obsoleta.

    public function enviaConversacion($chatMessages)
    {
        try {
            // Usar la clave de configuración de forma segura
            $apiKey = config('openai.key');

            if (empty($apiKey)) {
                throw new \Exception("OPENAI_API_KEY no está configurada en el archivo .env.");
            }

            $data = [
                "model" => "gpt-3.5-turbo",
                "messages" => $chatMessages,
                "temperature" => 0.7,
                "max_tokens" => 800,
                "n" => 1,
            ];

            $ch = curl_init('https://api.openai.com/v1/chat/completions');
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

            // Timeout para evitar cuelgues
            curl_setopt($ch, CURLOPT_TIMEOUT, 60);

            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'Authorization: Bearer ' . $apiKey
            ]);

            $response = curl_exec($ch);

            if (curl_errno($ch)) {
                $error_msg = curl_error($ch);
                curl_close($ch);
                throw new \Exception('cURL error: ' . $error_msg);
            }

            curl_close($ch);

            $responseArr = json_decode($response, true);

            if (isset($responseArr['error'])) {
                $error_msg = $responseArr['error']['message'] ?? 'Error desconocido de OpenAI.';
                throw new \Exception("OpenAI API Error: " . $error_msg);
            }

            if (isset($responseArr['choices']) && is_array($responseArr['choices']) && count($responseArr['choices']) > 0) {
                return $responseArr;
            } else {
                throw new \Exception('Invalid API response format (Empty choices)');
            }

        } catch (\Throwable $th) {
            \Log::error("ERROR EN ENVIA CONVERSACION:", ['msg' => $th->getMessage()]);
            throw $th;
        }
    }

    public function chatConversation(Request $request)
    {
        try {
            $prompt = $request->input('queryResult');
            $conversationId = $request->input('conversationId');
            $conversation = Conversation::find($conversationId);

            if (!$conversation) {
                
                $conversation = $this->createConversation(1);
                $conversationId = $conversation->id;
            }

            $this->createMessage($conversation, $prompt);

            $messagesForOpenAI = [
                ['role' => 'user', 'content' => $prompt]
            ];

            // Enviar a OpenAI
            $aiResponse = $this->enviaConversacion($messagesForOpenAI);

            // Guardar respuesta del Bot en BD
            $messageBot = Message::create([
                'conversation_id' => $conversationId,
                'api_id' => $aiResponse['id'] ?? null,
                'choices' => $aiResponse['choices'],
                'usage' => $aiResponse['usage']['total_tokens'] ?? null,
                'date_of_issue' => now(),
                'status' => 1,
                'emisor' => 'Gpt'
            ]);

            return response()->json([
                'response' => $aiResponse['choices'][0]['message']['content'],
                'chatMessages' => [['role' => 'assistant', 'content' => $aiResponse['choices'][0]['message']['content']]],
            ]);

        } catch (\Throwable $th) {
            Log::error("ERROR CRITICO EN IA:", [
                'mensaje' => $th->getMessage(),
                'archivo' => $th->getFile(),
                'linea' => $th->getLine()
            ]);

            return response()->json([
                'error' => 'Error de API',
                'message' => 'Hubo un problema al conectar con la inteligencia artificial.'
            ], 500);
        }
    }

    public function getChatHistory($conversationId)
    {
        $conversation = Conversation::with('messages')->find($conversationId);

        if (!$conversation) {
            return response()->json(['error' => 'Conversation not found'], 404);
        }

        $messages = $conversation->messages->map(function ($message) {
            return [
                'role' => $message->emisor === 'Gpt' ? 'assistant' : 'user',
                'content' => $message->choices[0]['text'] ?? null,
                'date_of_issue' => $message->date_of_issue,
            ];
        });

        return response()->json(['chatHistory' => $messages]);
    }

    public function getAllChatHistory()
    {
        $conversations = Conversation::with('messages')->get();

        $allChatHistory = $conversations->map(function ($conversation) {
            return [
                'conversationId' => $conversation->id,
                'messages' => $conversation->messages->map(function ($message) {
                    return [
                        'role' => $message->emisor === 'Gpt' ? 'assistant' : 'user',
                        'content' => $message->choices[0]['text'] ?? null,
                        'date_of_issue' => $message->date_of_issue,
                    ];
                }),
            ];
        });

        return response()->json(['allChatHistory' => $allChatHistory]);
    }
}